/*
 * project: RT-ADKmini_Firmware_v1.0
 * 概要:   RT-ADKmini用ファームウェア初期化関数群
 * File:   HardwareProfile - RTADKmini.h
 * 作者:   株式会社 アールティ
 */
#ifndef HARDWARE_PROFILE_PIC24FJ64GB002_PIM_H
#define HARDWARE_PROFILE_PIC24FJ64GB002_PIM_H

    #include <uart.h>
   
   
    #define DEMO_BOARD PIC24FJ64GB002_PIM
    #define EXPLORER_16
    #define PIC24FJ64GB002_PIM
    #define CLOCK_FREQ 32000000
    #define DEMO_BOARD_NAME_STRING "RT-ADKmini"

    //プロトタイピング
    void open_IO(WORD setting);
    void open_UART1(DWORD baudrate, BOOL stopbit, BYTE parity);
    void open_AD(void);
    void open_PWM0(void);
    void open_PWM1(void);
    void open_PWM2(void);
    void open_RS485(void);
    void open_normal_config(void);
    void open_geiger_config(void);
    void open_acc_config(void);
    void write_DOUT(WORD setting);
    WORD read_DIN(void);
    void write_UART1(char *buf,int len);
    void set_PWM0(WORD freq_count,WORD duty_count,BYTE division_ratio);
    void set_PWM1(WORD freq_count, BYTE duty_count, BYTE division_ratio);
    void set_PWM2(WORD freq_count, WORD duty_count, BYTE division_ratio);
    void set_PWM0_freq(unsigned long freq);
    void set_PWM1_freq(unsigned long freq);
    void set_PWM2_freq(unsigned long freq);
    void set_PWM0_duty(BYTE duty_count);
    void set_PWM1_duty(BYTE duty_count);
    void set_PWM2_duty(BYTE duty_count);
    void LEDDEBUG(char state);
    void LEDDEBUG_A(void);
    void LEDDEBUG_B(void);
    void LEDDEBUG_C(void);
    void CmdI2C(unsigned char data);
    void DataTrans(unsigned char sData);
    void GetDataI2C(unsigned char *Buffer, unsigned char CNT);
    void IdleI2C2(void);
    BYTE readRegister(BYTE address);

    /////
    void i2c2_init(void);
    void i2c2_start(void);
    void i2c2_repstart(void);
    void i2c2_stop(void);
    unsigned char i2c2_write(unsigned char);
    unsigned char i2c2_read(unsigned char);

    void MMA8452Active(void);
    void initMMA8452(BYTE fsr, BYTE dataRate);
    void MMA8452Standby(void);



    #define RS485_Send_EN   LATBbits.LATB15  //MAX485のEnable
    /////////////////////////////PWMのdutyの保存用変数/////////////////////////////
    float pwm0_duty;                   //COMMAND_PWMx_FREQを使った時にdutyを保持していないと
    float pwm1_duty;                   //dutyが不用意に書き換わってしまう
    float pwm2_duty;                   //あまり良い解決策ではない
    /////////////////////////////加速度センサの使用フラグ//////////////////////////
    BOOL accUseFlag       = FALSE;
    /////////////////////////////ピンの割り当て////////////////////////////////////////////
    // RT-ADKmini基板上の各種ピンとPICFJ64GB002マイコンのピンの対応
    //
    //
    // AIN0  = RA0 /RP5 /AN0 /CN2
    // AIN1  = RA1 /RP6 /AN1 /CN3
    // AIN2  = RB2 /RP2 /AN4 /CN6 /SDA2
    // AIN3  = RB3 /RP3 /AN5 /CN7 /SCL2
    //
    // DIN0  = RA2 /    /    /CN30 RT-ADKminiの基板上に実装されているタクトスイッチにつながっている
    // DIN1  = RA3 /    /    /CN29
    // DIN2  = RB4 /RP4 /    /CN1
    // DIN3  = RA4 /    /    /CN0　
    //
    // AOUT0 = RB0 /RP0 /AN2 /CN4
    // AOUT1 = RB1 /RP1 /AN3 /CN5
    // AOUT2 = RB15/RP15/AN9 /CN11 RS485を使うときはAOUT2はMAX485のENABLEなので使用できない
    //
    // DOUT0 = RB9 /RP9 /    /CN21 / SDA1 RT-ADKminiの基板上に実装されているLED1につながっている
    // DOUT1 = RB8 /RP8 /    /CN22 / SCL1
    // DOUT2 = RB7 /RP7 /    /CN23
    // DOUT3 = RB5 /    /    /CN27
    //
    // RX    = RB13/RP13/AN11/CN13
    // TX    = RB14/RP14/AN10/CN12
    //
    //////////////////////////////////////////////////////////////////////////////////

    /*----------------------------------------------------------------
    // 関数名 void open_IO(WORD setting)
    //
    // 引数 unsigned int setting
    //
    // 概要 各ピンの入力/出力の設定をする　　　　　　　　
    //      TRIS     ピンの入力/出力  1 = 入力  0 = 出力
    //      AD1PCFG  0-12bit: 0 = ANxピンはアナログ入力  1 = ANxピンはデジタル
    //      ※デフォルトではANxピンはアナログ入力になっている
    //
    //      settingの各bit  0 = 出力  1 = 入力
    //      0bit: AIN0  4bit: DIN0   8bit: AOUT0  12bit: DOUT0
    //      1bit: AIN1  5bit: DIN1   9bit: AOUT1  13bit: DOUT1
    //      2bit: AIN2  6bit: DIN2  10bit: AOUT2  14bit: DOUT2
    //      3bit: AIN3  7bit: DIN3  11bit:   X    15bit: DOUT3
    //
    // 注意 open_AD()より前で呼ぶこと
    //      そうでないとAD1PCFGの設定を上書きしてしまう
    //      他のモジュールの設定をする前にこの関数を呼ぶこと
    //
    //      タクトスイッチをつなげたときにチャタリングを防止するために
    //      ピンの状態読み出しは50msecに一回の頻度で行っている.
    ------------------------------------------------------------------*/
    void open_IO(WORD setting)
    {
        AD1PCFG = 0x1FFF;  //全ANピンをデジタルモードにする

        TRISAbits.TRISA0  = (setting & 0x0001) == 0x0001;
        TRISAbits.TRISA1  = (setting & 0x0002) == 0x0002;
        TRISBbits.TRISB2  = (setting & 0x0004) == 0x0004;
        TRISBbits.TRISB3  = (setting & 0x0008) == 0x0008;

        TRISAbits.TRISA2  = (setting & 0x0010) == 0x0010;
        TRISAbits.TRISA3  = (setting & 0x0020) == 0x0020;
        TRISBbits.TRISB4  = (setting & 0x0040) == 0x0040;
        TRISAbits.TRISA4  = (setting & 0x0080) == 0x0080;

        TRISBbits.TRISB0  = (setting & 0x0100) == 0x0100;
        TRISBbits.TRISB1  = (setting & 0x0200) == 0x0200;
        TRISBbits.TRISB15 = (setting & 0x0400) == 0x0400;
        //11bit目は未定義

        TRISBbits.TRISB9  = (setting & 0x1000) == 0x1000;
        TRISBbits.TRISB8  = (setting & 0x2000) == 0x2000;
        TRISBbits.TRISB7  = (setting & 0x4000) == 0x4000;
        TRISBbits.TRISB5  = (setting & 0x8000) == 0x8000;
    }
    
    /*----------------------------------------------------------------
    // 関数名 void open_UART1(DWORD baudrate, BOOL stopbit, BYTE parity)
    //
    // 引数 DWORD  baudrate  Min 15.3[baud] Max 1000000.0[baud]
    //
    //      BOOL stopbit     0   1[bit]
    //                       1   2[bit]
    //
    //      BYTE parity         data   parity
    //                       2  8[bit]  odd
    //                       1  8[bit]  even
    //                       0  8[bit]  なし
    //
    // 概要  RXピンとTXピンにUART1をセットする　　　　　　　　
    //      ボーレート,ストップビット数,パリティ,を設定可能
    //      データビット数は8bit,フロー制御なし
    //
    // 注意  実際のボーレートと真のボーレートとの間には誤差が生じることがある
    //       誤差が3%以下くらいならば,ほぼ正常に送受信可能
     ------------------------------------------------------------------*/
    void open_UART1(DWORD baudrate, BOOL stopbit, BYTE parity)
    {
        RPINR18bits.U1RXR = 13;     // UART1 RXをRP13にわりつけ
        RPOR7bits.RP14R = 3;        // UART1 TXをRP14にわりつけ

        U1MODE = 0;
        U1STA =  0;
        U1BRG = (int)((16000000.0/(16.0*(double)baudrate)) + 0.5) - 1;  //ボーレートの計算 0.5を足して四捨五入をしている
   
        switch(stopbit){                       //stopbitの設定
            case 0: U1MODEbits.PDSEL = 0; 
                break;
            case 1: U1MODEbits.PDSEL = 1;
                break;
        }
     
        U1MODEbits.PDSEL  = parity;            //parityの設定
        
        U1MODEbits.UARTEN = 1;                 //UART1有効化ビット
        U1MODEbits.RTSMD  = 1;                 //U1RTSは単方向モード
        U1STAbits.UTXEN   = 1;                 //UART1送信有効化ビット

    }
       
    /*----------------------------------------------------------------
    // 関数名 void open_AD(void)
    //
    // 引数 なし
    //
    // 概要 AIN0,1,2,3ピンをアナログインプットピンとして使用　　　　　　　　
    //    
    ------------------------------------------------------------------*/
    void open_AD(void)
    {
        //タイマ3
	PR3 = 6250/12-1;              //100m[sec]で12回AD変換が行われる
        T3CON = 0b1000000000110000;   //分周比 Pφ = 256  Tcy = 16M[Hz]
                                      //AD変換の間隔[sec] = (1/16M) * 256 * (PR3 + 1)
                                      //↑この間隔は1チャネル変換する毎の時間

        //ADC1の設定
        TRISAbits.TRISA0  = 1;       //AIN0,1,2,3は入力
        TRISAbits.TRISA1  = 1;
        TRISBbits.TRISB2  = 1;
        TRISBbits.TRISB3  = 1;

        AD1CON1 = 0x8044;            //タイマ3トリガ,サンプル自動
        AD1CON2 = 0x042C;            //AVdd,AVss,自動スキャン,12回目割り込み
        AD1CON3 = 0x1F05;
        AD1CHS  = 0x0000;
        AD1PCFG &= 0xFFCC;
        AD1CSSL = 0x0033;
        IEC0bits.AD1IE = 1;
      
    }
    
    /*----------------------------------------------------------------
    // 関数名 void open_PWM0(void)
    //
    // 引数 なし
    //
    // 概要 AOUT0ピンをPWM出力に設定　　　　　　　　
    //    
    ------------------------------------------------------------------*/
    void open_PWM0(void)
    {
        RPOR0bits.RP0R = 18;          //OC1をRP0(AOUT0)にわりつけ
        ///////タイマ5に関する設定///////
	T5CONbits.TCKPS = 0b11;       //FCY = 16MHz 分周比 256
                                      //分周比   11  256
	                              // 　　　  10   64
                                      // 　　　  01    8
                                      // 　　　  00    1
        T5CONbits.TON = 1;            //タイマ5スタート
        ///////PWMに関する設定//////////
        OC1R =  0;                    //PWMのduty設定
	OC1RS = 62500-1;              //PWMの周波数設定
	                              //周波数[Hz] = (FCY)/(分周比*(OC1RS+1))

        OC1CON1 = 0;
	OC1CON2 = 0;
	OC1CON2bits.SYNCSEL = 0x1f;   //同期要因をOCモジュール自身に設定
	OC1CON1bits.OCTSEL = 3;       //タイマ5を使用
	OC1CON1bits.OCM = 0b110;      //PWM動作モードを設定

    }

    /*----------------------------------------------------------------
    // 関数名 void open_PWM1(void)
    //
    // 引数 なし
    //
    // 概要 AOUT1ピンをPWM出力に設定　　　　　　　　
    //
    ------------------------------------------------------------------*/
    void open_PWM1(void)
    {
        RPOR0bits.RP1R = 19;          //OC2をRP1(AOUT1)にわりつけ
        ///////タイマ2に関する設定//////////
	T2CONbits.TCKPS = 0b11;       //FCY = 16MHz
                                      //分周比   11  256
	                              // 　　　  10   64
                                      // 　　　  01    8
                                      // 　　　  00    1
        T2CONbits.TON = 1;            //タイマ2スタート
        ///////PWMに関する設定//////////
        OC2R =  0;                    //PWMのduty設定
	OC2RS = 62500-1;                //PWMの周期の設定
	                              //周波数[Hz] = (FCY)/(分周比*(OC2RS+1))
        OC2CON1 = 0;
	OC2CON2 = 0;
	OC2CON2bits.SYNCSEL = 0x1f;   //同期要因をOCモジュール自身に設定
	OC2CON1bits.OCTSEL = 0;       //タイマ2を使用
	OC2CON1bits.OCM = 0b110;      //PWM動作モードを設定

    }

    /*----------------------------------------------------------------
    // 関数名 void open_PWM2(void)
    //
    // 引数 なし
    //
    // 概要 AOUT2ピンをPWM出力に設定　　　　　　　　
    //
    // 注意 RS485を使うときはPWM出力には使えない
    ------------------------------------------------------------------*/
    void open_PWM2(void)
    {
        RPOR7bits.RP15R = 20;         //OC3をRP15(AOUT2)にわりつけ
        ///////タイマ4に関する設定////////
	T4CONbits.TCKPS = 0b11;       //FCY = 16MHz 分周比 256
	                              //タイマ4スタート
                                      //分周比   11  256
	                              // 　　　  10   64
                                      // 　　　  01    8
                                      // 　　　  00    1
        T4CONbits.TON = 1;
        ///////PWMに関する設定//////////
        OC3R  = 0;                   //PWMのduty設定
        OC3RS = 62500-1;                //PWMの周期の設定
	                              //周波数[Hz] = (FCY)/(分周比*(OC3RS+1))

        OC3CON1 = 0;
	OC3CON2 = 0;
	OC3CON2bits.SYNCSEL = 0x1f; //同期要因をOCモジュール自身に設定
	OC3CON1bits.OCTSEL = 2;     //タイマ4を使用
	OC3CON1bits.OCM = 0b110;    //PWM動作モードを設定

    }

    /*----------------------------------------------------------------
    // 関数名 void open_RS485(void)
    //
    // 引数 なし
    //
    // 概要 AOUT2ピンをRS485のENABLEとして使用　　　　　　　
    //
    // 注意 この関数を呼んだ後にAOUT2ピンをデジタル出力以外として使用しないこと
    ------------------------------------------------------------------*/
    void open_RS485(void){
        AD1PCFGbits.PCFG9 = 1;  //AOUT2ピンをデジタルモードにする
        TRISBbits.TRISB15 = 0;
    }

    /*----------------------------------------------------------------
    // 関数名 void open_normal_config(void)
    //
    // 引数 なし
    //
    // 概要 DOUT0,1,2,3    デジタル出力
    // 　　 DIN0,1,2,3　   デジタル入力 　　　　　　
    // 　　 AOUT0,1,2      アナログ出力
    // 　　 AIN0,1,2,3     アナログ入力
    //      に設定
    ------------------------------------------------------------------*/
    void open_normal_config(void){
        open_IO(0x00F0);
        open_PWM0();
        open_PWM1();
        open_PWM2();
        open_AD();
       
    }

    /*----------------------------------------------------------------
    // 関数名 void open_geiger_config(void)
    //
    // 引数 なし
    //
    // 概要 poketガイガー type5を使用するときの設定を行う.
    // 　　 動作としては DIN1, DIN2をデジタル入力ピンとして
    // 　　 用いてタイマ割り込み2を用いてDIN1, DIN2ピンの状態を
    // 　　 監視し続け, 200msecの間に観測したDIN1ピンの立ち上がり
    //      回数をカウントしてそれをスマホ側に転送する.
    //      ただし, 1回の200msecループの間にDIN2ピンが立ち上がりノイズ
    //      を観測した場合はそのループでの放射線カウントを0にする.
    ------------------------------------------------------------------*/
    void open_geiger_config(void){
        //タイマ2初期化(タイマ割り込み1に使用)
        T2CON = 0b1000000000010000; //分周比1/8   16MHz
        //PR2 = 31250-1;	//500mSec
        PR2 = 10-1;	        //5microSec
        IPC1bits.T2IP = 6;	// 割り込みレベル 6
        IEC0bits.T2IE = 1; 	// Enable Int

        open_IO(0x00F0);
    }
     /*----------------------------------------------------------------
    // 関数名 void open_acc_config(void)
    //
    // 引数 なし
    //
    // 概要 i2cの三軸加速度センサMMA8452Qを使用するときの設定を行う.
    //      AIN2ピンがSDA, AIN3ピンがSCLとなる.  PICのピンアサイン名は
    //      SDA2,SCL2を使用
    // 　　
    ------------------------------------------------------------------*/
    void open_acc_config(void)
    {
        BYTE scale = 2;  // Sets full-scale range to +/-2, 4, or 8g. 加速度センサの加速度のスケール.
        BYTE dataRate = 0;  // 0=800Hz, 1=400, 2=200, 3=100, 4=50, 5=12.5, 6=6.25, 7=1.56 加速度センサのデータ取得周期
        BYTE c;
        //I2Cの初期設定////////////////////////////////////
        AD1PCFG = 0x1FFF;  //全ANピンをデジタルモードにする
        TRISBbits.TRISB2  = 1;  //AIN2ピンを入力に
        TRISBbits.TRISB3  = 1;  //AIN3ピンを入力に
        I2C2BRG = 0x13;        //100kHz@16MHz
        I2C2CON  =  0b1000000000000000;  //  7bit  address

        //MMA8452Qとの通信確立確認とデータ送信
        c = readRegister(0x0D);  // Read WHO_AM_I register
        if (c == 0x2A){
            accUseFlag = TRUE;
            LEDDEBUG_B();  //MMA8452Qとの通信に成功
            initMMA8452(scale, dataRate);
        }
    }







    /*----------------------------------------------------------------
    // 関数名 void write_DOUT(WORD setting)
    //
    // 引数 WORD setting
    //
    // 返り値 なし
    //
    // 概要  出力ピンに設定されているピンのHIGH/LOWを変える　　　　　　　　
    //      settingの各bit  0 = LOW  1 = HIGH
    //      0bit: AIN0  4bit: DIN0   8bit: AOUT0  12bit: DOUT0
    //      1bit: AIN1  5bit: DIN1   9bit: AOUT1  13bit: DOUT1
    //      2bit: AIN2  6bit: DIN2  10bit: AOUT2  14bit: DOUT2
    //      3bit: AIN3  7bit: DIN3  11bit:   X    15bit: DOUT3
    ------------------------------------------------------------------*/
    void write_DOUT(WORD setting)
    {
        LATAbits.LATA0 = (setting & 0x0001)  == 0x0001;
        LATAbits.LATA1 = (setting & 0x0002)  == 0x0002;
        LATBbits.LATB2 = (setting & 0x0004)  == 0x0004;
        LATBbits.LATB3 = (setting & 0x0008)  == 0x0008;

        LATAbits.LATA2 = (setting & 0x0010)  == 0x0010;
        LATAbits.LATA3 = (setting & 0x0020)  == 0x0020;
        LATBbits.LATB4 = (setting & 0x0040)  == 0x0040;
        LATAbits.LATA4 = (setting & 0x0080)  == 0x0080;

        LATBbits.LATB0 = (setting & 0x0100)  == 0x0100;
        LATBbits.LATB1 = (setting & 0x0200)  == 0x0200;
        LATBbits.LATB15 = (setting & 0x0400) == 0x0400;
        //11bit目は未定義

        LATBbits.LATB9 = (setting & 0x1000)  == 0x1000;
        LATBbits.LATB8 = (setting & 0x2000)  == 0x2000;
        LATBbits.LATB7 = (setting & 0x4000)  == 0x4000;
        LATBbits.LATB5 = (setting & 0x8000)  == 0x8000;
    }

    /*----------------------------------------------------------------
    // 関数名 WORD read_DIN(void)
    //
    // 引数  なし
    //
    // 概要  各ピンの状態を読みだす　　　　　　　　
    //      返り値の各bit  0 = LOW  1 = HIGH
    //      0bit: AIN0  4bit: DIN0   8bit: AOUT0  12bit: DOUT0
    //      1bit: AIN1  5bit: DIN1   9bit: AOUT1  13bit: DOUT1
    //      2bit: AIN2  6bit: DIN2  10bit: AOUT2  14bit: DOUT2
    //      3bit: AIN3  7bit: DIN3  11bit:   X    15bit: DOUT3
    ------------------------------------------------------------------*/
    WORD read_DIN(void)
    {
        WORD read = 0x0000;

        if(PORTAbits.RA0) read  |= 0x0001;
        if(PORTAbits.RA1) read  |= 0x0002;
        if(PORTBbits.RB2) read  |= 0x0004;
        if(PORTBbits.RB3) read  |= 0x0008;

        if(PORTAbits.RA2) read  |= 0x0010;
        if(PORTAbits.RA3) read  |= 0x0020;
        if(PORTBbits.RB4) read  |= 0x0040;
        if(PORTAbits.RA4) read  |= 0x0080;

        if(PORTBbits.RB0) read  |= 0x0100;
        if(PORTBbits.RB1) read  |= 0x0200;
        if(PORTBbits.RB15) read |= 0x0400;
        //11bit目は未定義

        if(PORTBbits.RB9) read  |= 0x1000;
        if(PORTBbits.RB8) read  |= 0x2000;
        if(PORTBbits.RB7) read  |= 0x4000;
        if(PORTBbits.RB5) read  |= 0x8000;

        return read;
    }

    /*----------------------------------------------------------------
    // 関数名 void write_UART1(char *buf, int len)
    //
    // 引数 char *buf, int len
    //
    // 概要 UART1でデータを送信する関数　　　　　　　　
    // 　　 第一引数で送信する配列を渡し,第二引数で配列の長さを渡す
    ------------------------------------------------------------------*/
    void write_UART1(char *buf,int len)
    {
            int c;
            RS485_Send_EN = 1;
            for(c = 0; c < len; c++){
                WriteUART1(buf[c]);
                while(!U1STAbits.TRMT);
            }
            RS485_Send_EN = 0;
           
    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM0(WORD palse_count, BYTE duty_count, BYTE division_ratio )
    //
    // 引数 WORD freq_count, WORD duty_count BYTE division ratio
    //                                            3: 256
    //                                            2: 64
    //                                            1: 8
    //                                            0: 1
    // 概要 PWM0の設定用関数　　　　　　　　
    //      FCY = 16M[Hz]
    // 　　 周波数[Hz] = (FCY)/(分周比*(palse_count+1))
    //      duty = duty_count/freq_count
    ------------------------------------------------------------------*/
    void set_PWM0(WORD freq_count,WORD duty_count,BYTE division_ratio)
    {
        T5CONbits.TCKPS = division_ratio;
        OC1RS = freq_count;
        OC1R  = duty_count;
    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM1(WORD freq_count, WORD duty_count BYTE division_ratio )
    //
    // 引数 WORD freq_count, BYTE duty_count  BYTE division ratio
    //                                             3: 256
    //                                             2: 64
    //                                             1: 8
    //                                             0: 1
    // 概要 PWM1の設定用関数　　　　　　　　
    //      FCY = 16M[Hz]
    // 　　 周波数[Hz] = (FCY)/(分周比*(palse_count+1))
    //      duty = duty_count/freq_count
     ------------------------------------------------------------------*/
    void set_PWM1(WORD freq_count, BYTE duty_count, BYTE division_ratio)
    {
        T2CONbits.TCKPS = division_ratio;
        OC2RS = freq_count;
        OC2R  = duty_count;
    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM2(WORD freq_count, WORD duty_count BYTE division_ratio )
    //
    // 引数 WORD freq_count, duty_count  BYTE division ratio
    //                                        3: 256
    //                                        2: 64
    //                                        1: 8
    //                                        0: 1
    // 概要 PWM2の設定用関数　　　　　　　　
    //      FCY = 16M[Hz]
    // 　　 周波数[Hz] = (FCY)/(分周比*(palse_count+1))
    //      duty = duty_count/freq_count
    ------------------------------------------------------------------*/
    void set_PWM2(WORD freq_count, WORD duty_count, BYTE division_ratio)
    {
        T4CONbits.TCKPS = division_ratio;
        OC3RS = freq_count;
        OC3R  = duty_count;
    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM0_freq(unsigned long freq)
    //
    // 引数 unsigned long freq
    //
    // 概要 PWM0の周波数の設定用関数　　　　　　　　
    // 　　 引数に設定したい周波数を渡す
    //      設定可能な周波数の範囲は1[Hz]～500k[Hz]
    //
    // 注意 細かく設定したいときはset_PWM0関数を使うこと
    // 　　 周波数が高くなるとPWMの分解能は小さくなる
    //      500k[Hz]のとき約32程の分解能
    ------------------------------------------------------------------*/
    void set_PWM0_freq(unsigned long freq)
    {
        int division_ratio; //分周比
       
        //PWMのdutyの分解能ができるだけ大きくなるように分周比を設定
        if(freq <= 4)
        {
            T5CONbits.TCKPS = 3;
            division_ratio  = 256;
        }
        if(freq > 4 && freq <= 32)
        {
            T5CONbits.TCKPS = 2;
            division_ratio  = 64;

        }
        if(freq > 32 && freq  <= 256)
        {
            T5CONbits.TCKPS = 1;
            division_ratio  = 8;
        }
        if(freq > 256)
        {
            T5CONbits.TCKPS = 0;
            division_ratio  = 1;
        }

        if(freq != 0) OC1RS = (unsigned int)(16000000.0/(division_ratio * freq)) - 1 ;
        else OC1RS = 0;      
        OC1R = OC1RS * pwm0_duty;
    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM1_freq(unsigned long freq)
    //
    // 引数 unsigned long freq
    //
    // 概要 PWM1の周波数の設定用関数　　　　　　　　
    // 　　 引数に設定したい周波数を渡す
    //      設定可能な周波数の範囲は1[Hz]～500k[Hz]
    //
    // 注意 細かく設定したいときはset_PWM1関数を使うこと
    // 　　 周波数が高くなるとPWMの分解能は小さくなる
    //      500k[Hz]のとき約32程の分解能
    ------------------------------------------------------------------*/

    void set_PWM1_freq(unsigned long freq)
    {
        int division_ratio; //分周比
     
        //PWMのdutyの分解能ができるだけ大きくなるように分周比を設定
        if(freq <= 4)
        {
            T2CONbits.TCKPS = 3;
            division_ratio  = 256;
        }
        if(freq > 4 && freq <= 32)
        {
            T2CONbits.TCKPS = 2;
            division_ratio  = 64;

        }
        if(freq > 32 && freq  <= 256)
        {
            T2CONbits.TCKPS = 1;
            division_ratio  = 8;

        }
        if(freq > 256)
        {
            T2CONbits.TCKPS = 0;
            division_ratio  = 1;
        }

        if(freq != 0) OC2RS = (16000000.0/(division_ratio * freq)) - 1 ;
        else OC2RS = 0;
        OC2R = OC2RS * pwm1_duty;


    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM2_freq(unsigned long freq)
    //
    // 引数 unsigned long freq
    //
    // 概要 PWM2の周波数の設定用関数　　　　　　　　
    // 　　 引数に設定したい周波数を渡す
    //      設定可能な周波数の範囲は1[Hz]～500k[Hz]
    //
    // 注意 細かく設定したいときはset_PWM2関数を使うこと
    // 　　 周波数が高くなるとPWMの分解能は小さくなる
    //      500k[Hz]のとき約32程の分解能
    ------------------------------------------------------------------*/

    void set_PWM2_freq(unsigned long freq)
    {
        int division_ratio; //分周比
        //PWMのdutyの分解能ができるだけ大きくなるように分周比を設定
        if(freq <= 4)
        {
            T4CONbits.TCKPS = 3;
            division_ratio  = 256;
        }
        if(freq > 4 && freq <= 32)
        {
            T4CONbits.TCKPS = 2;
            division_ratio  = 64;

        }
        if(freq > 32 && freq  <= 256)
        {
            T4CONbits.TCKPS = 1;
            division_ratio  = 8;

        }
        if(freq > 256)
        {
            T4CONbits.TCKPS = 0;
            division_ratio  = 1;
        }

        if(freq != 0)OC3RS = (16000000.0/(division_ratio * freq)) - 1 ;
        else OC3RS = 0;
        OC3R = OC3RS * pwm2_duty;
    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM0_duty(BYTE duty_count)
    //
    // 引数 BYTE duty_count
    //
    // 概要 PWM0のduty設定用関数　　　　　　　　
    // 　　 duty = duty_count/255
    ------------------------------------------------------------------*/
    void set_PWM0_duty(BYTE duty_count)
    {
        pwm0_duty =(duty_count/255.0);     //pwm0のduty設定値を保持
        OC1R = OC1RS * (duty_count/255.0);

    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM1_duty(BYTE duty_count)
    //
    // 引数 BYTE duty_count
    //
    // 概要 PWM1のduty設定用関数　　　　　　　　
    // 　　 duty = duty_count/255
    ------------------------------------------------------------------*/
    void set_PWM1_duty(BYTE duty_count)
    {
        pwm1_duty =(duty_count/255.0);     //pwm1のduty設定値を保持
        OC2R = OC2RS * (duty_count/255.0);
    }

    /*----------------------------------------------------------------
    // 関数名 void set_PWM2_duty(BYTE duty_count)
    //
    // 引数 BYTE duty_count
    //
    // 概要 PWM0のduty設定用関数　　　　　　　　
    // 　　 duty = duty_count/255
    ------------------------------------------------------------------*/
    void set_PWM2_duty(BYTE duty_count)
    {
        pwm2_duty =(duty_count/255.0);     //pwm2のduty設定値を保持
        OC3R = OC3RS * (duty_count/255.0);
    }


    /*----------------------------------------------------------------
    // 関数名 void LEDDEBUG(BOOL state);
    //
    // 引数 BOOL state
    //
    // 概要 RT-ADKmini基板上のLEDを制御する関数
    ------------------------------------------------------------------*/
    void LEDDEBUG(char state)
    {
        // DOUT0 = RB9 /RP9 /    /CN21 / SDA1 RT-ADKminiの基板上に実装されているLED1につながっている
        AD1PCFG = 0x1FFF;  //全ANピンをデジタルモードにする
        TRISBbits.TRISB9  = 0;
        LATBbits.LATB9 = state;
    }
    
    void LEDDEBUG_A(void)
    {
        long i;
        for(i=0; i< 250000; i++ ) LEDDEBUG(1);
        for(i=0; i< 250000; i++ ) LEDDEBUG(0);
        for(i=0; i< 250000; i++ ) LEDDEBUG(1);
        for(i=0; i< 250000; i++ ) LEDDEBUG(0);
        for(i=0; i< 250000; i++ ) LEDDEBUG(1);
        for(i=0; i< 250000; i++ ) LEDDEBUG(0);
        for(i=0; i< 250000; i++ ) LEDDEBUG(1);
        for(i=0; i< 250000; i++ ) LEDDEBUG(0);
    }

     void LEDDEBUG_B(void)
    {
        long i;
        for(i=0; i< 150000; i++ ) LEDDEBUG(1);
        for(i=0; i< 150000; i++ ) LEDDEBUG(0);
        for(i=0; i< 150000; i++ ) LEDDEBUG(1);
        for(i=0; i< 150000; i++ ) LEDDEBUG(0);
        for(i=0; i< 150000; i++ ) LEDDEBUG(1);
        for(i=0; i< 150000; i++ ) LEDDEBUG(0);
        for(i=0; i< 150000; i++ ) LEDDEBUG(1);
        for(i=0; i< 150000; i++ ) LEDDEBUG(0);
    }

     void LEDDEBUG_C(void)
    {
        long i;
        for(i=0; i< 50000; i++ ) LEDDEBUG(1);
        for(i=0; i< 50000; i++ ) LEDDEBUG(0);
        for(i=0; i< 50000; i++ ) LEDDEBUG(1);
        for(i=0; i< 50000; i++ ) LEDDEBUG(0);
        for(i=0; i< 50000; i++ ) LEDDEBUG(1);
        for(i=0; i< 50000; i++ ) LEDDEBUG(0);
        for(i=0; i< 50000; i++ ) LEDDEBUG(1);
        for(i=0; i< 50000; i++ ) LEDDEBUG(0);
    }


    /*----------------------------------------------------------------
    // 関数名 void readRegister(BYTE address)
    //
    // 引数  BYTE address
    //
    // 概要 I2Cで加速度センサmma8492のデータを1バイト読み出す関数
    ------------------------------------------------------------------*/
    BYTE readRegister(BYTE address)
    {
      BYTE data;
      
      i2c2_start();
      while(I2C2CONbits.SEN);
      
      DataTrans(0x3A);  
      DataTrans(address);
      
      i2c2_repstart();
      while(I2C2CONbits.RSEN);
      DataTrans(0x3B);

      I2C2CONbits.ACKDT =1;
      I2C2CONbits.RCEN =1;
      while(I2C2CONbits.RCEN);
      I2C2CONbits.ACKEN = 1;
      data = I2C2RCV;

      i2c2_stop();
      while(I2C2CONbits.PEN);   
      
      return data;
    }

    /*----------------------------------------------------------------
    // 関数名 void writeRegister(unsigned char address, unsigned char data)
    //
    // 引数  unsigned char address
    //       unsigned char data
    //
    // 概要 I2Cでmma8492Qに1バイト書き込みする関数
    ------------------------------------------------------------------*/
    void writeRegister(unsigned char address, unsigned char data)
    {
      i2c2_start();
      while(I2C2CONbits.SEN);

      DataTrans(0x3A);
      DataTrans(address);
      DataTrans(data);

      i2c2_stop();
      while(I2C2CONbits.PEN);
    }

    /*----------------------------------------------------------------
    // 関数名 void IdleI2C2(void)
    //
    // 引数 void
    //
    // 概要 I2Cでアイドル待ちする関数
    ------------------------------------------------------------------*/
    void IdleI2C2(void)
    {
        while(I2C2CONbits.SEN || I2C2CONbits.PEN || I2C2CONbits.RCEN || I2C2CONbits.ACKEN || I2C2STATbits.TRSTAT );
    }

    /*----------------------------------------------------------------
    // 関数名 void DataTrans(unsigned char sData)
    //
    // 引数 unsigned char sData
    //
    // 概要 1バイト送信関数
    ------------------------------------------------------------------*/
    void DataTrans(unsigned char sData)
    {
        I2C2TRN = sData;
        while(I2C2STATbits.TBF);
        while(I2C2STATbits.ACKSTAT);
        IdleI2C2();
    }



    void i2c2_idleck(void)
    {
        while((I2C2CON & 0x1F) | I2C2STATbits.TRSTAT);
    }

    void i2c2_start(void)
    {
        i2c2_idleck();
        I2C2CONbits.SEN = 1;
    }

    void i2c2_repstart(void)
    {
        i2c2_idleck();
        I2C2CONbits.RSEN = 1;
    }

    void i2c2_stop(void)
    {
       i2c2_idleck();
       I2C2CONbits.PEN = 1;
    }



       void initMMA8452(BYTE fsr, BYTE dataRate)
{
  MMA8452Standby();  // Must be in standby to change registers

  /* Set up the full scale range to 2, 4, or 8g. */
  if ((fsr==2)||(fsr==4)||(fsr==8))
    writeRegister(0x0E, fsr >> 2);
  else
    writeRegister(0x0E, 0);
  /* Setup the 3 data rate bits, from 0 to 7 */
  writeRegister(0x2A, readRegister(0x2A) & ~(0x38));
  if (dataRate <= 7)
    writeRegister(0x2A, readRegister(0x2A) | (dataRate << 3));
  /* Set up portrait/landscap registers */
  writeRegister(0x11, 0x40);  // Enable P/L
  writeRegister(0x13, 0x14);  // 29deg z-lock,
  writeRegister(0x14, 0x84);  // 45deg thresh, 14deg hyst
  writeRegister(0x12, 0x05);  // debounce counter at 100ms
  /* Set up single and double tap */
  writeRegister(0x21, 0x7F);  // enable single/double taps on all axes
  writeRegister(0x23, 0x20);  // x thresh at 2g
  writeRegister(0x24, 0x20);  // y thresh at 2g
  writeRegister(0x25, 0x20);  // z thresh at .5g
  writeRegister(0x26, 0x30);  // 60ms time limit, the min/max here is very dependent on output data rate
  writeRegister(0x27, 0x28);  // 200ms between taps min
  writeRegister(0x28, 0xFF);  // 1.275s (max value) between taps max
  /* Set up interrupt 1 and 2 */
  writeRegister(0x2C, 0x02);  // Active high, push-pull
  writeRegister(0x2D, 0x19);  // DRDY int enabled, P/L enabled
  writeRegister(0x2E, 0x01);  // DRDY on INT1, P/L on INT2

  MMA8452Active();  // Set to active to start reading
}

    /* Sets the MMA8452 to standby mode.
   It must be in standby to change most register settings */
    void MMA8452Standby(void)
{
  BYTE c = readRegister(0x2A);
  writeRegister(0x2A, c & ~(0x01));
}

/* Sets the MMA8452 to active mode.
   Needs to be in this mode to output data */
void MMA8452Active(void)
{
  BYTE c = readRegister(0x2A);
  writeRegister(0x2A, c | 0x01);
}




#endif	//HARDWARE_PROFILE_PIC24FJ64GB002_PIM_H
